/*
 MIT License

 Copyright (c) 2025 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.IO.Packaging;
using System.Net;
using System.Net.NetworkInformation;
using System.Net.Sockets;
using System.Numerics;
using System.Resources;
using System.Runtime.InteropServices;
using System.Threading;
using System.Windows.Forms;
using Windows.ApplicationModel.Background;

namespace HandyHarp2Client
{

    class Program
    {
        private const string mutexName = "jp.co.procyon-studio.handy-harp-2-client.windows.mutex";

        [STAThread]
        static void Main()
        {
            bool createdNew = false;
            Mutex? mutex = null;
            try
            {
                mutex = new Mutex(true, mutexName, out createdNew);
                if (createdNew)
                {
                    ApplicationConfiguration.Initialize();
                    Application.Run(new TrayApp());
                }
            }
            finally
            {
                if (createdNew && mutex != null)
                {
                    mutex.ReleaseMutex();
                }
            }
        }
    }

    class ClientData
    {
        public ClientData(TcpClient client)
        {
            this.Client = client;
            this.Stream = client.GetStream();
        }
        public TcpClient Client;
        public NetworkStream Stream;
    }

    class TrayApp : Form
    {
        [DllImport("user32.dll")]
        private static extern IntPtr GetForegroundWindow();

        [DllImport("user32.dll")]
        private static extern uint GetWindowThreadProcessId(IntPtr hWnd, out uint processId);

        public static string? GetActiveProcessName()
        {
            IntPtr hWnd = GetForegroundWindow();
            if (hWnd == IntPtr.Zero)
                return null;

            GetWindowThreadProcessId(hWnd, out uint pid);
            Process proc = Process.GetProcessById((int)pid);

            return proc?.MainModule?.ModuleName;
        }

        private NotifyIcon trayIcon;
        private ContextMenuStrip trayMenu;
        private Dictionary<string, ClientData> clients = new Dictionary<string, ClientData>();
        private System.Threading.Timer sendTimer;
        private List<ServiceBrowser> serviceBrowsers = new List<ServiceBrowser>();
        byte[] heartbeat = new byte[] { 0x00 };
        private AboutBox1? aboutBox1;

        public TrayApp()
        {
            // tH[Sɔ\
            this.ShowInTaskbar = false;
            this.WindowState = FormWindowState.Minimized;
            this.Visible = false;

            // ^XNgCACRƃj[
            trayMenu = new ContextMenuStrip();
            trayMenu.Items.Add("Status: Not Connected");
            trayMenu.Items.Add(new ToolStripSeparator());
            trayMenu.Items.Add("About", null, (s, e) =>
            {
                if (aboutBox1 == null || aboutBox1.IsDisposed)
                {
                    aboutBox1 = new AboutBox1();
                }
                aboutBox1.Show();
                aboutBox1.Activate();
            });
            trayMenu.Items.Add("Exit", null, (s, e) => quitApp());

            // gCACR
            trayIcon = new NotifyIcon()
            {
                Text = "Handy Harp II Client",
                Icon = Properties.Resources.icon1,
                ContextMenuStrip = trayMenu,
                Visible = true
            };

            trayIcon.MouseClick += new MouseEventHandler(trayIconMouseClick);

            // Bonjour T[rXT
            foreach (var ni in NetworkInterface.GetAllNetworkInterfaces())
            {
                var serviceBrowser = new ServiceBrowser();
                serviceBrowser.ServiceDiscovered += (sender, args) =>
                {
                    System.Diagnostics.Debug.WriteLine($"Found service {args.Service.Name}!");
                    ConnectToService(args.Service);
                };
                try
                {
                    serviceBrowser.Start("_handyharp2._tcp.local", ni);
                }
                catch {
                    continue;
                }
                serviceBrowsers.Add(serviceBrowser);
            }

            // 60bƂ0x00Mp^C}[
            sendTimer = new System.Threading.Timer(SendHeartbeat, null, 60000, 60000);

            // MĎ[vʃXbhŊJn
            Thread receiveThread = new Thread(ReceiveLoop);
            receiveThread.IsBackground = true;
            receiveThread.Start();
        }

        private void quitApp()
        {
            trayIcon.Visible = false;
            trayIcon.Dispose();
            Application.Exit();
        }
        private void trayIconMouseClick(object? sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                System.Reflection.MethodInfo? mi = typeof(NotifyIcon).GetMethod("ShowContextMenu", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                mi?.Invoke(trayIcon, null);
            }
        }
        private void UpdateStatus()
        {
            bool connected;
            lock (clients)
            {
                connected = clients.Count != 0;
            }
            if (connected)
            {
                trayIcon.Icon = Properties.Resources.icon2;
                trayMenu.Items[0].Text = "Status: Connected";
            }
            else
            {
                trayIcon.Icon = Properties.Resources.icon1;
                trayMenu.Items[0].Text = "Status: Not Connected";
            }
        }

        // ---------------------- ڑƉ ----------------------
        private void ConnectToService(NetworkService service)
        {
            if (service.IpAddresses == null || service.Port == null || service.Host == null) return;
            var addresses = service.IpAddresses;
            var port = service.Port;
            var host = service.Host;

            if (addresses.Length == 0)
            {
                System.Diagnostics.Debug.WriteLine($"No IP addresses.");
                return;
            }

            lock (clients)
            {
                {
                    ClientData? clientData;
                    if (clients.TryGetValue(host, out clientData) && clientData != null)
                    {
                        try
                        {
                            clientData.Stream.Write(heartbeat, 0, heartbeat.Length);
                            System.Diagnostics.Debug.WriteLine($"Connection to {host} is OK");
                            return;
                        }
                        catch
                        {
                            clientData.Client.Close();
                            clients.Remove(host);
                            System.Diagnostics.Debug.WriteLine($"Failed to send heartbeat to {host}");
                        }
                    }
                }
                foreach (var ip in addresses)
                {
                    try
                    {
                        TcpClient client = new TcpClient(ip.AddressFamily);
                        client.NoDelay = true;
                        client.Connect(ip, (int)port);
                        clients[host] = new ClientData(client);
                        System.Diagnostics.Debug.WriteLine($"Connected to {host} at {ip}:{port}");
                        break;
                    }
                    catch { continue; }
                }
            }
            BeginInvoke((MethodInvoker)delegate ()
            {
                UpdateStatus();
            });
        }

        // ---------------------- M[v ----------------------
        private void ReceiveLoop()
        {
            byte[] buffer = new byte[4];
            while (true)
            {
                Dictionary<string, ClientData> clients_temp;
                lock (clients) {
                    clients_temp = new Dictionary<string, ClientData>(clients);
                }
                
                var inactives = clients_temp.Where(kvp => !kvp.Value.Client.Connected).ToList();
                foreach (var kvp in clients_temp)
                {
                    var clientData = kvp.Value;
                    if (clientData.Client.Connected)
                    {
                        try
                        {
                            var stream = clientData.Stream;
                            var socket = stream.Socket;
                            if (socket.Poll(0, SelectMode.SelectRead))
                            {
                                int readBytes = 0;
                                if (socket.Available > 0)
                                {
                                    int read;
                                    do
                                    {
                                        read = stream.Read(buffer, 0, buffer.Length);
                                        if (read < 0)
                                        {
                                            readBytes = 0;
                                            break;
                                        }
                                        if (read == 4)
                                        {
                                            UInt32 command = BitConverter.ToUInt32(buffer, 0);
                                            BeginInvoke((MethodInvoker)delegate ()
                                            {
                                                ParseCommand(command);
                                            });
                                        }
                                        readBytes += read;
                                    } while (read != 0 && stream.DataAvailable);
                                    System.Diagnostics.Debug.WriteLine($"Received {readBytes} bytes from {kvp.Key}");
                                }
                                if (readBytes == 0)
                                {
                                    clientData.Client.Close();
                                    inactives.Add(kvp);
                                }
                            }
                        }
                        catch
                        {
                            System.Diagnostics.Debug.WriteLine($"Read({kvp.Key}) error");
                            clientData.Client.Close();
                            inactives.Add(kvp);
                        }
                    }
                }
                if (inactives.Count > 0)
                {
                    lock (clients)
                    {
                        foreach (var item in inactives)
                        {
                            clients.Remove(item.Key);
                        }
                    }
                    BeginInvoke((MethodInvoker)delegate ()
                    {
                        UpdateStatus();
                    });
                }

                Thread.Sleep(10);
            }
        }

        // ---------------------- 60bƂ̑M ----------------------
        private void SendHeartbeat(object? state)
        {
            Dictionary<string, ClientData> clients_temp;
            lock (clients)
            {
                clients_temp = new Dictionary<string, ClientData>(clients);
            }
            var inactives = clients.Where(kvp => !kvp.Value.Client.Connected).ToList();
            foreach (var kvp in clients)
            {
                var clientData = kvp.Value;
                if (clientData.Client.Connected)
                {
                    try
                    {
                        clientData.Stream.Write(heartbeat, 0, heartbeat.Length);
                        System.Diagnostics.Debug.WriteLine($"Sent heartbeat to {kvp.Key}");
                    }
                    catch
                    {
                        clientData.Client.Close();
                        inactives.Add(kvp);
                        System.Diagnostics.Debug.WriteLine($"Failed to send heartbeat to {kvp.Key}");
                    }
                }
            }
            if (inactives.Count > 0)
            {
                lock (clients)
                {
                    foreach (var item in inactives)
                    {
                        clients.Remove(item.Key);
                    }
                }
                BeginInvoke((MethodInvoker)delegate ()
                {
                    UpdateStatus();
                });
            }
        }

        private static /* async */ void ParseCommand(UInt32 command)
        {
            System.Diagnostics.Debug.WriteLine($"Command {command}");
            var currentAppExe = GetActiveProcessName()?.ToLower();
            if (currentAppExe == null) return;
            System.Diagnostics.Debug.WriteLine($"Current App: {currentAppExe}");

            switch (currentAppExe)
            {
                case "sibelius.exe":
                    switch (command & 0xff000000)
                    {
                        case 0x10000000:
                            List<InputSender.Input> inputs = new List<InputSender.Input>();
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_CONTROL, InputSender.KEYEVENTF_KEYDOWN));
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_T);
                            /*
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_CONTROL, InputSender.KEYEVENTF_KEYUP));
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_CONTROL, InputSender.KEYEVENTF_KEYDOWN));
                            */
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_MENU, InputSender.KEYEVENTF_KEYDOWN));

                            var func = (ref UInt32 command, ref List<InputSender.Input> inputs) =>
                            {
                                short vkey;
                                switch (command & 3)
                                {
                                    case 0:
                                        vkey = InputSender.VK_NUMPAD9;
                                        break;
                                    case 1:
                                    default:
                                        vkey = InputSender.VK_NUMPAD7;
                                        break;
                                    case 2:
                                        vkey = InputSender.VK_NUMPAD8;
                                        break;
                                }
                                InputSender.AddKeyPressToInputs(ref inputs, vkey);
                                command = command >> 2;
                            };

                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_ADD);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_MENU, InputSender.KEYEVENTF_KEYUP));
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_CONTROL, InputSender.KEYEVENTF_KEYUP));
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_ESCAPE);
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_ESCAPE);

                            InputSender.SendInput(inputs.ToArray());
                            break;
                    }
                    break;

                case "dorico6.exe":
                case "dorico7.exe":
                case "dorico8.exe":
                case "dorico9.exe":
                    switch (command & 0xff000000)
                    {
                        case 0x10000000:
                            List<InputSender.Input> inputs = new List<InputSender.Input>();
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_SHIFT, InputSender.KEYEVENTF_KEYDOWN));
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_P);
                            inputs.Add(InputSender.KeyInputV(InputSender.VK_SHIFT, InputSender.KEYEVENTF_KEYUP));
                            /*
                            InputSender.SendInput(inputs.ToArray());

                            await Task.Delay(200);

                            inputs.Clear();
                            */
                            var func = (ref UInt32 command, ref List<InputSender.Input> inputs) =>
                            {
                                string str;
                                switch (command & 3)
                                {
                                    case 0:
                                        str = "^";
                                        break;
                                    case 1:
                                    default:
                                        str = "-";
                                        break;
                                    case 2:
                                        str = "v";
                                        break;
                                }
                                InputSender.AddKeyPressToInputs(ref inputs, str);
                                command = command >> 2;
                            };
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            func(ref command, ref inputs);
                            InputSender.AddKeyPressToInputs(ref inputs, InputSender.VK_RETURN);

                            InputSender.SendInput(inputs.ToArray());
                            break;
                    }
                    break;
            }
        }
    }

}