﻿/*
 MIT License

 Copyright (c) 2025 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

using System;
using System.Drawing;
using System.Runtime.InteropServices;

namespace HandyHarp2Client
{
    public static class InputSender
    {
        public static void KeyPress(string text)
        {

            Input[] inputs = new Input[2 * text.Length];
            for (var i_char = 0; i_char < text.Length; i_char++)
            {
                var val_sc = (short)text[i_char];
                var i_in = 2 * i_char;
                inputs[i_in] = KeyInputS(val_sc, KEYEVENTF_KEYDOWN | KEYEVENTF_UNICODE);
                inputs[i_in+1] = KeyInputS(val_sc, KEYEVENTF_KEYUP | KEYEVENTF_UNICODE);
            }
            InputSender.SendInput(inputs);
        }

        public static void KeyPress(char text) => InputSender.KeyPress(text.ToString());

        public static void KeyPress(short[] vkeys)
        {

            var inputs = new Input[2 * vkeys.Length];
            for (var i_vkey = 0; i_vkey < vkeys.Length; i_vkey++)
            {

                var val_vk = vkeys[i_vkey];
                var i_in = 2 * i_vkey;
                inputs[i_in] = KeyInputV(val_vk, KEYEVENTF_KEYDOWN);

                i_in = 2 * i_vkey + 1;
                inputs[i_in] = KeyInputV(val_vk, KEYEVENTF_KEYUP);
            }
            InputSender.SendInput(inputs);
        }

        public static void AddKeyPressToInputs(ref List<Input> inputs, string text)
        {
            for (var i_char = 0; i_char < text.Length; i_char++)
            {

                var val_sc = (short)text[i_char];
                inputs.Add(KeyInputS(val_sc, KEYEVENTF_KEYDOWN | KEYEVENTF_UNICODE));
                inputs.Add(KeyInputS(val_sc, KEYEVENTF_KEYUP | KEYEVENTF_UNICODE));
            }

        }

        public static void AddKeyPressToInputs(ref List<Input> inputs, short vkey)
        {
            inputs.Add(KeyInputV(vkey, KEYEVENTF_KEYDOWN));
            inputs.Add(KeyInputV(vkey, KEYEVENTF_KEYUP));
        }


        public static Input KeyInputS(short scanCode, int flags)
        {
            var input = new Input();
            input.Type = INPUT_KEYBOARD;
            input.ui.Keyboard.ScanCode = scanCode;
            input.ui.Keyboard.Flags = flags;
            return input;
        }
        public static Input KeyInputV(short vkey, int flags)
        {
            var input = new Input();
            input.Type = INPUT_KEYBOARD;
            input.ui.Keyboard.VirtualKey = vkey;
            input.ui.Keyboard.Flags = flags;
            return input;
        }

        public static void KeyPress(short vkey) => InputSender.KeyPress(new[] { vkey });

        // https://wisdom.sakura.ne.jp/system/winapi/win32/win32.html
        public const short VK_F = 0x46;
        public const short VK_P = 0x50;
        public const short VK_T = 0x54;

        public const short VK_RETURN = 0x0D; // Enter
        public const short VK_SHIFT = 0x10;
        public const short VK_CONTROL = 0x11;
        public const short VK_MENU = 0x12; // Alt
        public const short VK_ESCAPE = 0x1B;

        public const short VK_NUMPAD7 = 0x67;
        public const short VK_NUMPAD8 = 0x68;
        public const short VK_NUMPAD9 = 0x69;
        public const short VK_ADD = 0x6B; // KeyPad
        
        public static void SendInput(Input[] inputs)
        {
            SendInput(inputs.Length, inputs, Marshal.SizeOf<Input>());
        }

        [DllImport("user32.dll", SetLastError = true)]
        public extern static void SendInput(int nInputs, Input[] pInputs, int cbsize);

        public const int INPUT_MOUSE = 0;
        public const int INPUT_KEYBOARD = 1;
        public const int INPUT_HARDWARE = 2;

        public const int MOUSEEVENTF_MOVE = 0x1;
        public const int MOUSEEVENTF_ABSOLUTE = 0x8000;
        public const int MOUSEEVENTF_LEFTDOWN = 0x2;
        public const int MOUSEEVENTF_LEFTUP = 0x4;
        public const int MOUSEEVENTF_RIGHTDOWN = 0x8;
        public const int MOUSEEVENTF_RIGHTUP = 0x10;
        public const int MOUSEEVENTF_MIDDLEDOWN = 0x20;
        public const int MOUSEEVENTF_MIDDLEUP = 0x40;
        public const int MOUSEEVENTF_WHEEL = 0x800;
        public const int WHEEL_DELTA = 120;

        public const int KEYEVENTF_EXTENDEDKEY = 0x0001;
        public const int KEYEVENTF_KEYDOWN = 0x0000;
        public const int KEYEVENTF_KEYUP = 0x0002;
        public const int KEYEVENTF_SCANCODE = 0x0008;
        public const int KEYEVENTF_UNICODE = 0x0004;

        [StructLayout(LayoutKind.Sequential)]
        public struct MouseInput
        {
            public int X;
            public int Y;
            public int Data;
            public int Flags;
            public int Time;
            public IntPtr ExtraInfo;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct KeyboardInput
        {
            public short VirtualKey;
            public short ScanCode;
            public int Flags;
            public int Time;
            public IntPtr ExtraInfo;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct HardwareInput
        {
            public int uMsg;
            public short wParamL;
            public short wParamH;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct Input
        {
            public int Type;
            public InputUnion ui;
        }

        [StructLayout(LayoutKind.Explicit)]
        public struct InputUnion
        {
            [FieldOffset(0)]
            public MouseInput Mouse;
            [FieldOffset(0)]
            public KeyboardInput Keyboard;
            [FieldOffset(0)]
            public HardwareInput Hardware;
        }
    }
}